<?php
session_start();

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

require_once '../api/config.php';
$conn = getDBConnection();

// Ensure static_pages table exists
try {
    $conn->exec("CREATE TABLE IF NOT EXISTS static_pages (
        id INT AUTO_INCREMENT PRIMARY KEY,
        slug VARCHAR(50) UNIQUE NOT NULL,
        title VARCHAR(150) NOT NULL,
        content LONGTEXT NOT NULL,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_slug (slug)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
    
    // Insert default pages if they don't exist
    $defaultPages = [
        ['about', 'আমাদের সম্পর্কে', 'এই পেজে আপনার সম্পর্কে তথ্য যোগ করুন।'],
        ['terms', 'শর্তাবলী', 'ব্যবহারের শর্তাবলী এখানে লিখুন।'],
        ['privacy', 'গোপনীয়তা নীতি', 'গোপনীয়তা নীতি এখানে লিখুন।']
    ];
    
    foreach ($defaultPages as $page) {
        $stmt = $conn->prepare("INSERT IGNORE INTO static_pages (slug, title, content) VALUES (?, ?, ?)");
        $stmt->execute($page);
    }
} catch (Exception $e) {
    // ignore
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $slug = trim($_POST['slug'] ?? '');
    $title = trim($_POST['title'] ?? '');
    $content = trim($_POST['content'] ?? '');

    if (!empty($slug) && !empty($title) && !empty($content)) {
        $stmt = $conn->prepare("INSERT INTO static_pages (slug, title, content) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE title = ?, content = ?");
        $stmt->execute([$slug, $title, $content, $title, $content]);
        header('Location: pages.php?saved=1&slug=' . urlencode($slug));
        exit;
    }
}

// Get all pages
$pages = $conn->query("SELECT * FROM static_pages ORDER BY slug")->fetchAll();

// Get page to edit (if slug provided)
$editPage = null;
if (isset($_GET['edit'])) {
    $slug = $_GET['edit'];
    $stmt = $conn->prepare("SELECT * FROM static_pages WHERE slug = ?");
    $stmt->execute([$slug]);
    $editPage = $stmt->fetch();
}

?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Static Pages - Admin Panel</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
    <style>
        .sidebar {
            min-height: 100vh;
            background: #2c3e50;
        }
        .sidebar a {
            color: #ecf0f1;
            text-decoration: none;
            padding: 15px;
            display: block;
        }
        .sidebar a:hover {
            background: #34495e;
        }
        .sidebar a.active {
            background: #3498db;
        }
        #content {
            min-height: 400px;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-2 sidebar p-0">
                <div class="p-3">
                    <h4 class="text-white">Mathbaria Shop</h4>
                    <p class="text-white-50 small">Admin Panel</p>
                </div>
                <nav>
                    <a href="index.php"><i class="bi bi-house"></i> Dashboard</a>
                    <a href="products.php"><i class="bi bi-box"></i> Products</a>
                    <a href="pending.php"><i class="bi bi-clock-history"></i> Pending Approval</a>
                    <a href="reports.php"><i class="bi bi-flag"></i> Reported Ads</a>
                    <a href="users.php"><i class="bi bi-people"></i> Users</a>
                    <a href="settings.php"><i class="bi bi-gear"></i> Settings</a>
                    <a href="pages.php" class="active"><i class="bi bi-file-text"></i> Pages</a>
                    <a href="logout.php"><i class="bi bi-box-arrow-right"></i> Logout</a>
                </nav>
            </div>

            <!-- Main Content -->
            <div class="col-md-10 p-4">
                <h2>Static Pages Management</h2>
                <hr>

                <?php if (isset($_GET['saved'])): ?>
                    <div class="alert alert-success">Page saved successfully.</div>
                <?php endif; ?>

                <div class="row">
                    <!-- Pages List -->
                    <div class="col-md-4">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">Pages</h5>
                            </div>
                            <div class="card-body p-0">
                                <div class="list-group list-group-flush">
                                    <?php foreach ($pages as $page): ?>
                                        <a href="?edit=<?php echo urlencode($page['slug']); ?>" 
                                           class="list-group-item list-group-item-action <?php echo ($editPage && $editPage['slug'] === $page['slug']) ? 'active' : ''; ?>">
                                            <strong><?php echo htmlspecialchars($page['title']); ?></strong>
                                            <br>
                                            <small class="text-muted">/<?php echo htmlspecialchars($page['slug']); ?></small>
                                            <br>
                                            <small class="text-muted">Updated: <?php echo date('d M Y', strtotime($page['updated_at'])); ?></small>
                                        </a>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Edit Form -->
                    <div class="col-md-8">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">
                                    <?php echo $editPage ? 'Edit Page' : 'Select a page to edit'; ?>
                                </h5>
                            </div>
                            <div class="card-body">
                                <?php if ($editPage): ?>
                                    <form method="POST">
                                        <input type="hidden" name="slug" value="<?php echo htmlspecialchars($editPage['slug']); ?>">
                                        
                                        <div class="mb-3">
                                            <label for="title" class="form-label">Page Title</label>
                                            <input type="text" class="form-control" id="title" name="title" 
                                                   value="<?php echo htmlspecialchars($editPage['title']); ?>" required>
                                        </div>

                                        <div class="mb-3">
                                            <label for="slug" class="form-label">Slug (URL)</label>
                                            <input type="text" class="form-control" id="slug" 
                                                   value="<?php echo htmlspecialchars($editPage['slug']); ?>" 
                                                   disabled>
                                            <small class="form-text text-muted">Slug cannot be changed</small>
                                        </div>

                                        <div class="mb-3">
                                            <label for="content" class="form-label">Content</label>
                                            <textarea class="form-control" id="content" name="content" rows="15" required><?php echo htmlspecialchars($editPage['content']); ?></textarea>
                                            <small class="form-text text-muted">You can use HTML tags for formatting</small>
                                        </div>

                                        <div class="mt-4">
                                            <button type="submit" class="btn btn-primary">
                                                <i class="bi bi-save"></i> Save Page
                                            </button>
                                            <a href="pages.php" class="btn btn-secondary">Cancel</a>
                                        </div>
                                    </form>
                                <?php else: ?>
                                    <p class="text-muted">Select a page from the list to edit its content.</p>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>
</html>


