<?php
session_start();

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

require_once '../api/config.php';
$conn = getDBConnection();

// Handle single / bulk actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $ids = [];

    // Approve all filtered pending products
    if ($action === 'approve_filtered') {
        // Build filters from current GET query
        $where = [];
        $params = [];
        $search = trim($_GET['search'] ?? '');
        $statusFilter = $_GET['status'] ?? '';
        $unionFilter = trim($_GET['union'] ?? '');
        $categoryFilter = trim($_GET['category'] ?? '');
        $dateFrom = trim($_GET['date_from'] ?? '');
        $dateTo = trim($_GET['date_to'] ?? '');

        if ($search !== '') {
            $where[] = "(title LIKE ? OR description LIKE ?)";
            $params[] = '%' . $search . '%';
            $params[] = '%' . $search . '%';
        }
        if ($unionFilter !== '') {
            $where[] = "union_name = ?";
            $params[] = $unionFilter;
        }
        if ($categoryFilter !== '') {
            $where[] = "category = ?";
            $params[] = $categoryFilter;
        }
        if ($dateFrom !== '') {
            $where[] = "DATE(created_at) >= ?";
            $params[] = $dateFrom;
        }
        if ($dateTo !== '') {
            $where[] = "DATE(created_at) <= ?";
            $params[] = $dateTo;
        }
        // Force pending status to avoid approving non-pending
        $where[] = "status = 'pending'";

        $whereSql = '';
        if (!empty($where)) {
            $whereSql = 'WHERE ' . implode(' AND ', $where);
        }

        // Select IDs to approve
        $selectSql = "SELECT id FROM products $whereSql ORDER BY created_at DESC";
        $stmtSel = $conn->prepare($selectSql);
        $stmtSel->execute($params);
        $rows = $stmtSel->fetchAll();
        $ids = array_map(fn($r) => (int)$r['id'], $rows);

        if (!empty($ids)) {
            $placeholders = implode(',', array_fill(0, count($ids), '?'));
            $paramsUpdate = $ids;
            array_unshift($paramsUpdate, 'approved', $_SESSION['admin_id']);
            $stmtUp = $conn->prepare(
                "UPDATE products 
                 SET status = ?, approved_by = ?, approved_at = NOW() 
                 WHERE id IN ($placeholders)"
            );
            $stmtUp->execute($paramsUpdate);
            header('Location: products.php?updated=1&approved_count=' . count($ids));
            exit;
        } else {
            header('Location: products.php?updated=0');
            exit;
        }
    }

    // Single row action
    if (!empty($_POST['product_id'])) {
        $ids[] = (int)$_POST['product_id'];
    }
    // Bulk action (checkboxes)
    if (!empty($_POST['selected']) && is_array($_POST['selected'])) {
        foreach ($_POST['selected'] as $id) {
            $ids[] = (int)$id;
        }
    }

    $ids = array_values(array_unique(array_filter($ids)));

    if (!empty($ids) && in_array($action, ['approve', 'reject', 'delete'], true)) {
        $placeholders = implode(',', array_fill(0, count($ids), '?'));

        if ($action === 'delete') {
            // Hard delete
            $stmt = $conn->prepare("DELETE FROM products WHERE id IN ($placeholders)");
            $stmt->execute($ids);
            header('Location: products.php?deleted=1');
            exit;
        } else {
            // Status update (approve / reject)
            $status = $action === 'approve' ? 'approved' : 'rejected';
            $params = $ids;
            array_unshift($params, $status, $_SESSION['admin_id']);
            $stmt = $conn->prepare(
                "UPDATE products 
                 SET status = ?, approved_by = ?, approved_at = NOW() 
                 WHERE id IN ($placeholders)"
            );
            $stmt->execute($params);
            header('Location: products.php?updated=1');
            exit;
        }
    }
}

// Filters
$where = [];
$params = [];

$search = trim($_GET['search'] ?? '');
$statusFilter = $_GET['status'] ?? '';
$unionFilter = trim($_GET['union'] ?? '');
$categoryFilter = trim($_GET['category'] ?? '');
$dateFrom = trim($_GET['date_from'] ?? '');
$dateTo = trim($_GET['date_to'] ?? '');

if ($search !== '') {
    $where[] = "(p.title LIKE ? OR p.description LIKE ?)";
    $params[] = '%' . $search . '%';
    $params[] = '%' . $search . '%';
}

if ($statusFilter !== '' && in_array($statusFilter, ['pending', 'approved', 'rejected'], true)) {
    $where[] = "p.status = ?";
    $params[] = $statusFilter;
}

if ($unionFilter !== '') {
    $where[] = "p.union_name = ?";
    $params[] = $unionFilter;
}

if ($categoryFilter !== '') {
    $where[] = "p.category = ?";
    $params[] = $categoryFilter;
}

if ($dateFrom !== '') {
    $where[] = "DATE(p.created_at) >= ?";
    $params[] = $dateFrom;
}

if ($dateTo !== '') {
    $where[] = "DATE(p.created_at) <= ?";
    $params[] = $dateTo;
}

$whereSql = '';
if (!empty($where)) {
    $whereSql = 'WHERE ' . implode(' AND ', $where);
}

// Fetch distinct unions & categories for filter dropdowns
$unions = $conn->query("SELECT DISTINCT union_name FROM products ORDER BY union_name")->fetchAll(PDO::FETCH_COLUMN);
$categories = $conn->query("SELECT DISTINCT category FROM products ORDER BY category")->fetchAll(PDO::FETCH_COLUMN);

// Fetch products with filters
$sql = "
    SELECT p.*, u.name AS seller_name, u.phone AS seller_phone 
    FROM products p 
    LEFT JOIN users u ON p.seller_id = u.id 
    $whereSql
    ORDER BY p.created_at DESC
";
$stmt = $conn->prepare($sql);
$stmt->execute($params);
$products = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Products - Admin Panel</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
    <style>
        .sidebar {
            min-height: 100vh;
            background: #2c3e50;
        }
        .sidebar a {
            color: #ecf0f1;
            text-decoration: none;
            padding: 15px;
            display: block;
        }
        .sidebar a:hover {
            background: #34495e;
        }
        .sidebar a.active {
            background: #3498db;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-2 sidebar p-0">
                <div class="p-3">
                    <h4 class="text-white">Mathbaria Shop</h4>
                    <p class="text-white-50 small">Admin Panel</p>
                </div>
                <nav>
                    <a href="index.php"><i class="bi bi-house"></i> Dashboard</a>
                    <a href="products.php" class="active"><i class="bi bi-box"></i> Products</a>
                    <a href="pending.php"><i class="bi bi-clock-history"></i> Pending Approval</a>
                    <a href="reports.php"><i class="bi bi-flag"></i> Reported Ads</a>
                    <a href="users.php"><i class="bi bi-people"></i> Users</a>
                    <a href="settings.php"><i class="bi bi-gear"></i> Settings</a>
                    <a href="pages.php"><i class="bi bi-file-text"></i> Pages</a>
                    <a href="logout.php"><i class="bi bi-box-arrow-right"></i> Logout</a>
                </nav>
            </div>

            <!-- Main Content -->
            <div class="col-md-10 p-4">
                <h2>All Products</h2>
                <hr>

                <?php if (isset($_GET['deleted'])): ?>
                    <div class="alert alert-success">Product deleted successfully.</div>
                <?php elseif (isset($_GET['updated'])): ?>
                    <div class="alert alert-success">Product status updated successfully.</div>
                <?php endif; ?>

                <div class="card">
                    <div class="card-header">
                        <div class="d-flex justify-content-between align-items-center flex-wrap gap-2">
                            <h5 class="mb-0">Products List (<?php echo count($products); ?>)</h5>
                            <?php 
                                $pendingCount = 0;
                                foreach ($products as $pp) { if (($pp['status'] ?? '') === 'pending') { $pendingCount++; } }
                            ?>
                            <form class="d-flex flex-wrap gap-2" method="GET" action="products.php">
                                <input type="text" name="search" class="form-control form-control-sm" 
                                       placeholder="Search title/description"
                                       value="<?php echo htmlspecialchars($search); ?>">

                                <select name="status" class="form-select form-select-sm">
                                    <option value="">All Status</option>
                                    <option value="pending" <?php echo $statusFilter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                                    <option value="approved" <?php echo $statusFilter === 'approved' ? 'selected' : ''; ?>>Approved</option>
                                    <option value="rejected" <?php echo $statusFilter === 'rejected' ? 'selected' : ''; ?>>Rejected</option>
                                </select>

                                <select name="union" class="form-select form-select-sm">
                                    <option value="">All Unions</option>
                                    <?php foreach ($unions as $uName): ?>
                                        <option value="<?php echo htmlspecialchars($uName); ?>"
                                            <?php echo $unionFilter === $uName ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($uName); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>

                                <select name="category" class="form-select form-select-sm">
                                    <option value="">All Categories</option>
                                    <?php foreach ($categories as $cName): ?>
                                        <option value="<?php echo htmlspecialchars($cName); ?>"
                                            <?php echo $categoryFilter === $cName ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($cName); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>

                                <input type="date" name="date_from" class="form-control form-control-sm"
                                       value="<?php echo htmlspecialchars($dateFrom); ?>">
                                <input type="date" name="date_to" class="form-control form-control-sm"
                                       value="<?php echo htmlspecialchars($dateTo); ?>">

                                <button type="submit" class="btn btn-sm btn-primary">
                                    Filter
                                </button>
                                <a href="products.php" class="btn btn-sm btn-secondary">
                                    Reset
                                </a>
                            </form>
                            <form method="POST" onsubmit="return confirm('Approve all PENDING products in current filter?');">
                                <input type="hidden" name="action" value="approve_filtered">
                                <button type="submit" class="btn btn-sm btn-success" <?php echo $pendingCount === 0 ? 'disabled' : ''; ?>>
                                    <i class="bi bi-check2-circle"></i>
                                    Approve all pending<?php echo $pendingCount > 0 ? " ($pendingCount)" : ""; ?>
                                </button>
                            </form>
                        </div>
                    </div>
                    <div class="card-body">
                        <?php if (empty($products)): ?>
                            <p class="text-muted">No products found.</p>
                        <?php else: ?>
                            <form method="POST" onsubmit="return confirm('Are you sure you want to perform this action?');">
                                <div class="d-flex mb-2 gap-2 align-items-center flex-wrap">
                                    <select name="action" class="form-select form-select-sm" required style="max-width: 180px;">
                                        <option value="">Bulk action...</option>
                                        <option value="approve">Approve selected</option>
                                        <option value="reject">Reject selected</option>
                                        <option value="delete">Delete selected</option>
                                    </select>
                                    <button type="submit" class="btn btn-sm btn-outline-primary">
                                        Apply
                                    </button>
                                </div>

                                <div class="table-responsive">
                                    <table class="table table-hover align-middle">
                                        <thead>
                                            <tr>
                                                <th><input type="checkbox" onclick="
                                                    const cbs = document.querySelectorAll('.product-checkbox');
                                                    cbs.forEach(cb => cb.checked = this.checked);
                                                "></th>
                                                <th>ID</th>
                                                <th>Title</th>
                                                <th>Price</th>
                                                <th>Category</th>
                                                <th>Union</th>
                                                <th>Status</th>
                                                <th>Seller</th>
                                                <th>Created</th>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($products as $p): ?>
                                                <tr>
                                                    <td>
                                                        <input type="checkbox" class="product-checkbox" 
                                                               name="selected[]" value="<?php echo $p['id']; ?>">
                                                    </td>
                                                    <td><?php echo $p['id']; ?></td>
                                                    <td><?php echo htmlspecialchars($p['title']); ?></td>
                                                    <td>৳<?php echo number_format($p['price'], 2); ?></td>
                                                    <td><?php echo htmlspecialchars($p['category']); ?></td>
                                                    <td><?php echo htmlspecialchars($p['union_name']); ?></td>
                                                    <td>
                                                        <span class="badge bg-<?php
                                                            echo $p['status'] === 'approved' ? 'success' :
                                                                ($p['status'] === 'pending' ? 'warning text-dark' : 'secondary');
                                                        ?>">
                                                            <?php echo htmlspecialchars(ucfirst($p['status'])); ?>
                                                        </span>
                                                    </td>
                                                    <td>
                                                        <?php echo htmlspecialchars($p['seller_name'] ?? 'N/A'); ?>
                                                        <?php if (!empty($p['seller_phone'])): ?>
                                                            <br><small class="text-muted"><?php echo htmlspecialchars($p['seller_phone']); ?></small>
                                                        <?php endif; ?>
                                                    </td>
                                                    <td><?php echo date('d M Y', strtotime($p['created_at'])); ?></td>
                                                    <td>
                                                        <a href="pending.php?id=<?php echo $p['id']; ?>" class="btn btn-sm btn-primary mb-1">
                                                            View
                                                        </a>
                                                        <form method="POST" style="display:inline-block">
                                                            <input type="hidden" name="product_id" value="<?php echo $p['id']; ?>">
                                                            <button type="submit" name="action" value="delete" class="btn btn-sm btn-danger">
                                                                Delete
                                                            </button>
                                                        </form>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </form>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>
</html>


