<?php
require_once '../config.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(false, 'Invalid request method', null, 405);
}

// Get user ID from request (should be sent from app after login)
$userId = $_POST['seller_id'] ?? null;
if (!$userId) {
    sendResponse(false, 'User authentication required', null, 401);
}

// Validate required fields
$required = ['title', 'description', 'price', 'category', 'condition', 'union'];
foreach ($required as $field) {
    if (empty($_POST[$field])) {
        sendResponse(false, "$field is required", null, 400);
    }
}

$title = trim($_POST['title']);
$description = trim($_POST['description']);
$price = floatval($_POST['price']);
$category = trim($_POST['category']);
$condition = trim($_POST['condition']);
$union = trim($_POST['union']);

// Validate union (FINAL - Bengali names only)
$validUnions = ['মঠবাড়িয়া', 'আমড়াগাছিয়া', 'বেতমোর রাজপাড়া', 'বড়মাছুয়া', 'দাউদখালী', 
                'ধানীসাফা', 'গুলিশাখালী', 'মিরুখালী', 'শাপলেজা', 'টিকিকাটা', 'তুষখালী'];
if (!in_array($union, $validUnions)) {
    sendResponse(false, 'Invalid union', null, 400);
}

// Validate category (FINAL - Bengali names only - MVP)
$validCategories = ['মোবাইল ও ইলেকট্রনিক্স', 'মোটরসাইকেল ও যানবাহন', 'আসবাবপত্র', 
                    'গরু, ছাগল ও পশুপাখি', 'বাড়ি ভাড়া / জমি', 'সার্ভিস / সেবা', 
                    'পোশাক ও ব্যক্তিগত জিনিসপত্র', 'বই ও শিক্ষা সামগ্রী', 'অন্যান্য'];
if (!in_array($category, $validCategories)) {
    sendResponse(false, 'Invalid category', null, 400);
}

// Get seller phone
try {
    $conn = getDBConnection();
    
    // Check user and blocked status
    $colCheck = $conn->query("SHOW COLUMNS FROM users LIKE 'is_blocked'");
    $hasBlockedColumn = $colCheck && $colCheck->fetch();

    if ($hasBlockedColumn) {
        $stmt = $conn->prepare("SELECT phone, is_blocked FROM users WHERE id = ?");
    } else {
    $stmt = $conn->prepare("SELECT phone FROM users WHERE id = ?");
    }
    $stmt->execute([$userId]);
    $user = $stmt->fetch();
    
    if (!$user) {
        sendResponse(false, 'User not found', null, 404);
    }

    if ($hasBlockedColumn && !empty($user['is_blocked'])) {
        sendResponse(false, 'আপনার একাউন্ট ব্লক করা হয়েছে। নতুন প্রোডাক্ট পোস্ট করা যাবে না।', null, 403);
    }
    
    $sellerPhone = $user['phone'];
    
    // Insert product
    $stmt = $conn->prepare("INSERT INTO products (title, description, price, category, condition_type, union_name, seller_id, seller_phone, status) 
                           VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'pending')");
    $stmt->execute([$title, $description, $price, $category, $condition, $union, $userId, $sellerPhone]);
    
    $productId = $conn->lastInsertId();
    
    // Handle image uploads
    $uploadedImages = [];
    $uploadErrors = [];
    
    // Check if upload directory exists and is writable
    // UPLOAD_DIR is now absolute path from config.php
    $uploadDirPath = UPLOAD_DIR;
    if (!file_exists($uploadDirPath)) {
        if (!mkdir($uploadDirPath, 0755, true)) {
            error_log("Failed to create upload directory: " . $uploadDirPath);
            sendResponse(false, 'Upload directory creation failed: ' . $uploadDirPath, null, 500);
        }
    }
    
    if (!is_writable($uploadDirPath)) {
        error_log("Upload directory is not writable: " . $uploadDirPath);
        sendResponse(false, 'Upload directory is not writable: ' . $uploadDirPath, null, 500);
    }
    
    // Debug: Log what we received
    error_log("=== PRODUCT UPLOAD DEBUG START ===");
    error_log("FILES received: " . print_r($_FILES, true));
    error_log("POST received: " . print_r($_POST, true));
    error_log("Upload directory: " . $uploadDirPath);
    error_log("Product ID: " . $productId);
    
    // Check for images in different possible formats
    $filesReceived = false;
    $imageCount = 0;
    
    // Format 1: images[] (array format from Flutter)
    if (isset($_FILES['images']) && is_array($_FILES['images']['name'])) {
        $filesReceived = true;
        $imageCount = count($_FILES['images']['name']);
    }
    // Format 2: images (single file or array)
    elseif (isset($_FILES['images'])) {
        $filesReceived = true;
        if (is_array($_FILES['images']['name'])) {
        $imageCount = count($_FILES['images']['name']);
        } else {
            $imageCount = 1;
        }
    }
    
    if (!$filesReceived) {
        error_log("No files received in request");
        // Don't fail if no images - product can be created without images
        sendResponse(true, 'Product added successfully (no images uploaded). Waiting for admin approval.', [
            'product_id' => $productId,
            'images' => [],
            'warning' => 'No images were uploaded'
        ], 201);
    }
        
        if ($imageCount > 5) {
            // Delete product if too many images
            $conn->prepare("DELETE FROM products WHERE id = ?")->execute([$productId]);
            sendResponse(false, 'Maximum 5 images allowed', null, 400);
        }
        
    // Process images
        for ($i = 0; $i < $imageCount; $i++) {
        $errorCode = $_FILES['images']['error'][$i] ?? UPLOAD_ERR_NO_FILE;
        
        if ($errorCode === UPLOAD_ERR_OK) {
                $file = [
                    'name' => $_FILES['images']['name'][$i],
                    'type' => $_FILES['images']['type'][$i],
                    'tmp_name' => $_FILES['images']['tmp_name'][$i],
                    'size' => $_FILES['images']['size'][$i]
                ];
                
            // Validate file exists
            if (!file_exists($file['tmp_name'])) {
                $uploadErrors[] = "Image $i: Temporary file not found";
                error_log("Image $i: Temporary file not found: " . $file['tmp_name']);
                continue;
            }
            
            // Validate image type - check both MIME type and file extension
            $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            $allowedExtensions = ['jpg', 'jpeg', 'png', 'webp'];
            $mimeType = $file['type'];
            
            // If MIME type is application/octet-stream or empty, detect from file content
            if ($mimeType === 'application/octet-stream' || empty($mimeType)) {
                // Try to detect actual MIME type from file content
                if (function_exists('finfo_open')) {
                    $finfo = finfo_open(FILEINFO_MIME_TYPE);
                    $detectedMime = finfo_file($finfo, $file['tmp_name']);
                    finfo_close($finfo);
                    if ($detectedMime && in_array($detectedMime, ALLOWED_IMAGE_TYPES)) {
                        $mimeType = $detectedMime; // Update with detected MIME type
                    }
                }
            }
            
            // Validate: check MIME type OR file extension
            $isValidMime = in_array($mimeType, ALLOWED_IMAGE_TYPES);
            $isValidExtension = in_array($extension, $allowedExtensions);
            
            if (!$isValidMime && !$isValidExtension) {
                $uploadErrors[] = "Image $i: Invalid type (MIME: {$mimeType}, Extension: {$extension})";
                error_log("Image $i: Invalid type - MIME: {$mimeType}, Extension: {$extension}");
                continue;
            }
            
            // If MIME type was invalid but extension is valid, use extension-based MIME type
            if (!$isValidMime && $isValidExtension) {
                $mimeTypeMap = [
                    'jpg' => 'image/jpeg',
                    'jpeg' => 'image/jpeg',
                    'png' => 'image/png',
                    'webp' => 'image/webp'
                ];
                $mimeType = $mimeTypeMap[$extension] ?? 'image/jpeg';
            }
                
            // Validate file size
            if ($file['size'] > MAX_IMAGE_SIZE) {
                $uploadErrors[] = "Image $i: File too large ({$file['size']} bytes)";
                error_log("Image $i: File too large: " . $file['size']);
                continue;
            }
            
            // Generate unique filename (extension already defined above)
            if (empty($extension)) {
                // Try to detect extension from MIME type
                $mimeToExt = [
                    'image/jpeg' => 'jpg',
                    'image/jpg' => 'jpg',
                    'image/png' => 'png',
                    'image/webp' => 'webp'
                ];
                $extension = $mimeToExt[$mimeType] ?? 'jpg';
            }
                $filename = $productId . '_' . $i . '_' . time() . '.' . $extension;
            $filepath = $uploadDirPath . '/' . $filename;
                
                // Upload file
            $moveResult = move_uploaded_file($file['tmp_name'], $filepath);
            error_log("Image $i move_uploaded_file result: " . ($moveResult ? 'SUCCESS' : 'FAILED') . " - from: {$file['tmp_name']} to: $filepath");
            
            if ($moveResult) {
                // Verify file was actually written
                if (file_exists($filepath) && filesize($filepath) > 0) {
                    // Insert image record into database
                    try {
                    $stmt = $conn->prepare("INSERT INTO product_images (product_id, image_path, image_order) VALUES (?, ?, ?)");
                        $insertResult = $stmt->execute([$productId, $filename, $i]);
                        
                        if ($insertResult) {
                            $insertedId = $conn->lastInsertId();
                            error_log("Image $i database insert SUCCESS - ID: $insertedId, filename: $filename");
                            
                            // Verify the record was actually inserted
                            $verifyStmt = $conn->prepare("SELECT id FROM product_images WHERE id = ?");
                            $verifyStmt->execute([$insertedId]);
                            $verifyResult = $verifyStmt->fetch();
                            
                            if ($verifyResult) {
                                // Build full URL for response
                                $imageUrl = rtrim(UPLOAD_URL, '/') . '/' . $filename;
                                $uploadedImages[] = $imageUrl;
                                error_log("Image $i uploaded and saved successfully: $filename");
                            } else {
                                $uploadErrors[] = "Image $i: Database record not found after insert";
                                error_log("Image $i: Database record verification FAILED - ID: $insertedId");
            }
                        } else {
                            $uploadErrors[] = "Image $i: Database insert failed";
                            error_log("Image $i: Database insert FAILED - filename: $filename");
                        }
                    } catch (PDOException $e) {
                        $uploadErrors[] = "Image $i: Database error - " . $e->getMessage();
                        error_log("Image $i: Database exception - " . $e->getMessage());
                    }
                } else {
                    $uploadErrors[] = "Image $i: File write verification failed";
                    error_log("Image $i: File write verification FAILED - filepath: $filepath, exists: " . (file_exists($filepath) ? 'yes' : 'no'));
                }
            } else {
                $uploadErrors[] = "Image $i: move_uploaded_file failed";
                error_log("Image $i: move_uploaded_file FAILED - tmp: {$file['tmp_name']}, dest: $filepath, tmp_exists: " . (file_exists($file['tmp_name']) ? 'yes' : 'no'));
            }
        } else {
            // Handle upload errors
            $errorMessages = [
                UPLOAD_ERR_INI_SIZE => 'File exceeds upload_max_filesize',
                UPLOAD_ERR_FORM_SIZE => 'File exceeds MAX_FILE_SIZE',
                UPLOAD_ERR_PARTIAL => 'File was only partially uploaded',
                UPLOAD_ERR_NO_FILE => 'No file was uploaded',
                UPLOAD_ERR_NO_TMP_DIR => 'Missing temporary folder',
                UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
                UPLOAD_ERR_EXTENSION => 'A PHP extension stopped the file upload'
            ];
            $errorMsg = $errorMessages[$errorCode] ?? "Unknown error ($errorCode)";
            $uploadErrors[] = "Image $i: $errorMsg";
            error_log("Image $i upload error: $errorMsg (code: $errorCode)");
        }
    }
    
    // Log summary
    error_log("=== UPLOAD SUMMARY ===");
    error_log("Total images attempted: $imageCount");
    error_log("Successfully uploaded: " . count($uploadedImages));
    error_log("Errors: " . count($uploadErrors));
    if (!empty($uploadErrors)) {
        error_log("Error details: " . print_r($uploadErrors, true));
    }
    error_log("=== PRODUCT UPLOAD DEBUG END ===");
    
    $responseMessage = 'Product added successfully. Waiting for admin approval.';
    if (!empty($uploadErrors)) {
        $responseMessage .= ' Some images failed to upload: ' . implode(', ', $uploadErrors);
    }
    if (empty($uploadedImages) && !empty($uploadErrors)) {
        $responseMessage = 'Product added but no images were uploaded. Errors: ' . implode(', ', $uploadErrors);
    }
    
    sendResponse(true, $responseMessage, [
        'product_id' => $productId,
        'images' => $uploadedImages,
        'uploaded_count' => count($uploadedImages),
        'errors' => $uploadErrors
    ], 201);
    
} catch(PDOException $e) {
    sendResponse(false, 'Failed to add product: ' . $e->getMessage(), null, 500);
}

?>

