<?php
// Upload Folders Check and Setup Script
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../api/config.php';

echo "<!DOCTYPE html>";
echo "<html lang='bn'>";
echo "<head>";
echo "<meta charset='UTF-8'>";
echo "<meta name='viewport' content='width=device-width, initial-scale=1.0'>";
echo "<title>Upload Folders Check - Mathbaria Shop</title>";
echo "<link href='https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css' rel='stylesheet'>";
echo "<style>body { padding: 20px; background: #f5f5f5; }</style>";
echo "</head>";
echo "<body>";
echo "<div class='container'>";
echo "<div class='card shadow'>";
echo "<div class='card-body p-4'>";
echo "<h2 class='mb-4'>আপলোড ফোল্ডার চেক করছি...</h2>";
echo "<hr>";

$baseDir = dirname(__DIR__);
$uploadDir = $baseDir . '/uploads';
$productsDir = $uploadDir . '/products';

$issues = [];
$fixed = [];

// Check and create uploads directory
echo "<h4>১. uploads ফোল্ডার চেক করছি...</h4>";
if (!is_dir($uploadDir)) {
    if (mkdir($uploadDir, 0755, true)) {
        echo "<div class='alert alert-success'>✅ 'uploads' ফোল্ডার তৈরি করা হয়েছে!</div>";
        $fixed[] = "uploads folder created";
    } else {
        echo "<div class='alert alert-danger'>❌ 'uploads' ফোল্ডার তৈরি করতে পারিনি!</div>";
        $issues[] = "Cannot create uploads folder";
    }
} else {
    echo "<div class='alert alert-success'>✅ 'uploads' ফোল্ডার আছে!</div>";
    
    // Check write permission
    if (!is_writable($uploadDir)) {
        echo "<div class='alert alert-warning'>⚠️ 'uploads' ফোল্ডারে write permission নেই!</div>";
        $issues[] = "uploads folder not writable";
    } else {
        echo "<div class='alert alert-success'>✅ 'uploads' ফোল্ডারে write permission আছে!</div>";
    }
}

// Check and create products directory
echo "<h4>২. uploads/products ফোল্ডার চেক করছি...</h4>";
if (!is_dir($productsDir)) {
    if (mkdir($productsDir, 0755, true)) {
        echo "<div class='alert alert-success'>✅ 'uploads/products' ফোল্ডার তৈরি করা হয়েছে!</div>";
        $fixed[] = "products folder created";
    } else {
        echo "<div class='alert alert-danger'>❌ 'uploads/products' ফোল্ডার তৈরি করতে পারিনি!</div>";
        $issues[] = "Cannot create products folder";
    }
} else {
    echo "<div class='alert alert-success'>✅ 'uploads/products' ফোল্ডার আছে!</div>";
    
    // Check write permission
    if (!is_writable($productsDir)) {
        echo "<div class='alert alert-warning'>⚠️ 'uploads/products' ফোল্ডারে write permission নেই!</div>";
        $issues[] = "products folder not writable";
    } else {
        echo "<div class='alert alert-success'>✅ 'uploads/products' ফোল্ডারে write permission আছে!</div>";
    }
}

// Create .htaccess file for security
echo "<h4>৩. Security ফাইল (.htaccess) তৈরি করছি...</h4>";
$htaccessFile = $productsDir . '/.htaccess';
$htaccessContent = "# Prevent direct PHP execution\n";
$htaccessContent .= "<Files *.php>\n";
$htaccessContent .= "    Order Deny,Allow\n";
$htaccessContent .= "    Deny from all\n";
$htaccessContent .= "</Files>\n\n";
$htaccessContent .= "# Allow image files\n";
$htaccessContent .= "<FilesMatch \"\\.(jpg|jpeg|png|gif|webp)$\">\n";
$htaccessContent .= "    Order Allow,Deny\n";
$htaccessContent .= "    Allow from all\n";
$htaccessContent .= "</FilesMatch>\n\n";
$htaccessContent .= "# Prevent directory listing\n";
$htaccessContent .= "Options -Indexes\n\n";
$htaccessContent .= "# Set cache headers for images\n";
$htaccessContent .= "<IfModule mod_expires.c>\n";
$htaccessContent .= "    ExpiresActive On\n";
$htaccessContent .= "    ExpiresByType image/jpeg \"access plus 1 year\"\n";
$htaccessContent .= "    ExpiresByType image/png \"access plus 1 year\"\n";
$htaccessContent .= "    ExpiresByType image/gif \"access plus 1 year\"\n";
$htaccessContent .= "    ExpiresByType image/webp \"access plus 1 year\"\n";
$htaccessContent .= "</IfModule>\n";

if (!file_exists($htaccessFile)) {
    if (file_put_contents($htaccessFile, $htaccessContent)) {
        echo "<div class='alert alert-success'>✅ '.htaccess' ফাইল তৈরি করা হয়েছে!</div>";
        $fixed[] = ".htaccess file created";
    } else {
        echo "<div class='alert alert-warning'>⚠️ '.htaccess' ফাইল তৈরি করতে পারিনি!</div>";
        $issues[] = "Cannot create .htaccess file";
    }
} else {
    echo "<div class='alert alert-info'>ℹ️ '.htaccess' ফাইল ইতিমধ্যে আছে!</div>";
}

// Create index.php to prevent directory listing
echo "<h4>৪. index.php ফাইল তৈরি করছি...</h4>";
$indexFile = $productsDir . '/index.php';
$indexContent = "<?php\n";
$indexContent .= "// Prevent directory listing\n";
$indexContent .= "header('HTTP/1.0 403 Forbidden');\n";
$indexContent .= "exit('Access denied');\n";
$indexContent .= "?>\n";

if (!file_exists($indexFile)) {
    if (file_put_contents($indexFile, $indexContent)) {
        echo "<div class='alert alert-success'>✅ 'index.php' ফাইল তৈরি করা হয়েছে!</div>";
        $fixed[] = "index.php file created";
    } else {
        echo "<div class='alert alert-warning'>⚠️ 'index.php' ফাইল তৈরি করতে পারিনি!</div>";
        $issues[] = "Cannot create index.php file";
    }
} else {
    echo "<div class='alert alert-info'>ℹ️ 'index.php' ফাইল ইতিমধ্যে আছে!</div>";
}

// Check UPLOAD_DIR constant
echo "<h4>৫. Configuration চেক করছি...</h4>";
$configUploadDir = dirname(__DIR__) . '/' . str_replace('../', '', UPLOAD_DIR);
echo "<div class='alert alert-info'>";
echo "<strong>Config UPLOAD_DIR:</strong> " . htmlspecialchars(UPLOAD_DIR) . "<br>";
echo "<strong>Absolute Path:</strong> " . htmlspecialchars($configUploadDir) . "<br>";
echo "<strong>UPLOAD_URL:</strong> " . htmlspecialchars(UPLOAD_URL) . "<br>";
echo "</div>";

if (realpath($configUploadDir) === realpath($productsDir)) {
    echo "<div class='alert alert-success'>✅ Configuration path সঠিক!</div>";
} else {
    echo "<div class='alert alert-warning'>⚠️ Configuration path মিলছে না!</div>";
    $issues[] = "Configuration path mismatch";
}

// Test write permission
echo "<h4>৬. Write Permission টেস্ট করছি...</h4>";
$testFile = $productsDir . '/test_write_' . time() . '.txt';
if (file_put_contents($testFile, 'test')) {
    unlink($testFile);
    echo "<div class='alert alert-success'>✅ Write permission কাজ করছে! ফাইল লিখতে এবং মুছতে পারছি!</div>";
} else {
    echo "<div class='alert alert-danger'>❌ Write permission কাজ করছে না! ফাইল লিখতে পারছি না!</div>";
    $issues[] = "Write permission test failed";
}

// List existing files
echo "<h4>৭. বিদ্যমান ফাইলগুলো দেখছি...</h4>";
$files = glob($productsDir . '/*');
$imageFiles = array_filter($files, function($file) {
    $ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));
    return in_array($ext, ['jpg', 'jpeg', 'png', 'gif', 'webp']);
});

$fileCount = count($imageFiles);
if ($fileCount > 0) {
    echo "<div class='alert alert-info'>";
    echo "মোট <strong>{$fileCount}টি</strong> ছবি ফাইল পাওয়া গেছে!<br>";
    echo "<small>(প্রথম 10টি দেখানো হচ্ছে)</small><br><br>";
    echo "<ul>";
    $shown = 0;
    foreach ($imageFiles as $file) {
        if ($shown++ >= 10) break;
        $filename = basename($file);
        $size = filesize($file);
        $sizeKB = round($size / 1024, 2);
        echo "<li>{$filename} ({$sizeKB} KB)</li>";
    }
    if ($fileCount > 10) {
        echo "<li>... এবং আরো " . ($fileCount - 10) . "টি ফাইল</li>";
    }
    echo "</ul>";
    echo "</div>";
} else {
    echo "<div class='alert alert-info'>ℹ️ এখনো কোনো ছবি আপলোড করা হয়নি!</div>";
}

// Summary
echo "<hr>";
echo "<h3>সারাংশ</h3>";

if (empty($issues)) {
    echo "<div class='alert alert-success'><h4>✅ সবকিছু ঠিক আছে!</h4>";
    echo "<p>আপলোড ফোল্ডারগুলো সঠিকভাবে সেটআপ করা আছে এবং প্রোডাক্ট ছবি আপলোড করার জন্য প্রস্তুত!</p>";
    if (!empty($fixed)) {
        echo "<p><strong>যা করা হয়েছে:</strong></p><ul>";
        foreach ($fixed as $item) {
            echo "<li>" . htmlspecialchars($item) . "</li>";
        }
        echo "</ul>";
    }
    echo "</div>";
} else {
    echo "<div class='alert alert-warning'><h4>⚠️ কিছু সমস্যা পাওয়া গেছে:</h4><ul>";
    foreach ($issues as $issue) {
        echo "<li>" . htmlspecialchars($issue) . "</li>";
    }
    echo "</ul></div>";
}

echo "<p><a href='index.php' class='btn btn-primary'>এডমিন প্যানেলে ফিরে যান</a></p>";

echo "</div>";
echo "</div>";
echo "</div>";
echo "</body>";
echo "</html>";
?>

