<?php
require_once '../config.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(false, 'Invalid request method', null, 405);
}

// Get user ID from request (should be sent from app after login)
$userId = $_POST['seller_id'] ?? null;
if (!$userId) {
    sendResponse(false, 'User authentication required', null, 401);
}

// Validate required fields
$required = ['title', 'description', 'price', 'category', 'condition', 'union'];
foreach ($required as $field) {
    if (empty($_POST[$field])) {
        sendResponse(false, "$field is required", null, 400);
    }
}

$title = trim($_POST['title']);
$description = trim($_POST['description']);
$price = floatval($_POST['price']);
$category = trim($_POST['category']);
$condition = trim($_POST['condition']);
$union = trim($_POST['union']);

// Validate union (FINAL - Bengali names only)
$validUnions = ['মঠবাড়িয়া', 'আমড়াগাছিয়া', 'বেতমোর রাজপাড়া', 'বড়মাছুয়া', 'দাউদখালী', 
                'ধানীসাফা', 'গুলিশাখালী', 'মিরুখালী', 'শাপলেজা', 'টিকিকাটা', 'তুষখালী'];
if (!in_array($union, $validUnions)) {
    sendResponse(false, 'Invalid union', null, 400);
}

// Validate category (FINAL - Bengali names only - MVP)
$validCategories = ['মোবাইল ও ইলেকট্রনিক্স', 'মোটরসাইকেল ও যানবাহন', 'আসবাবপত্র', 
                    'গরু, ছাগল ও পশুপাখি', 'বাড়ি ভাড়া / জমি', 'সার্ভিস / সেবা', 
                    'পোশাক ও ব্যক্তিগত জিনিসপত্র', 'বই ও শিক্ষা সামগ্রী', 'অন্যান্য'];
if (!in_array($category, $validCategories)) {
    sendResponse(false, 'Invalid category', null, 400);
}

// Get seller phone
try {
    $conn = getDBConnection();
    
    $stmt = $conn->prepare("SELECT phone FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch();
    
    if (!$user) {
        sendResponse(false, 'User not found', null, 404);
    }
    
    $sellerPhone = $user['phone'];
    
    // Insert product
    $stmt = $conn->prepare("INSERT INTO products (title, description, price, category, condition_type, union_name, seller_id, seller_phone, status) 
                           VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'pending')");
    $stmt->execute([$title, $description, $price, $category, $condition, $union, $userId, $sellerPhone]);
    
    $productId = $conn->lastInsertId();
    
    // Handle image uploads
    $uploadedImages = [];
    if (isset($_FILES['images']) && is_array($_FILES['images']['name'])) {
        $imageCount = count($_FILES['images']['name']);
        
        if ($imageCount > 5) {
            // Delete product if too many images
            $conn->prepare("DELETE FROM products WHERE id = ?")->execute([$productId]);
            sendResponse(false, 'Maximum 5 images allowed', null, 400);
        }
        
        for ($i = 0; $i < $imageCount; $i++) {
            if ($_FILES['images']['error'][$i] === UPLOAD_ERR_OK) {
                $file = [
                    'name' => $_FILES['images']['name'][$i],
                    'type' => $_FILES['images']['type'][$i],
                    'tmp_name' => $_FILES['images']['tmp_name'][$i],
                    'size' => $_FILES['images']['size'][$i]
                ];
                
                // Validate image
                if (!in_array($file['type'], ALLOWED_IMAGE_TYPES)) {
                    continue;
                }
                
                if ($file['size'] > MAX_IMAGE_SIZE) {
                    continue;
                }
                
                // Generate unique filename
                $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
                $filename = $productId . '_' . $i . '_' . time() . '.' . $extension;
                $filepath = UPLOAD_DIR . $filename;
                
                // Upload file
                if (move_uploaded_file($file['tmp_name'], $filepath)) {
                    // Insert image record
                    $stmt = $conn->prepare("INSERT INTO product_images (product_id, image_path, image_order) VALUES (?, ?, ?)");
                    $stmt->execute([$productId, $filename, $i]);
                    $uploadedImages[] = UPLOAD_URL . $filename;
                }
            }
        }
    }
    
    sendResponse(true, 'Product added successfully. Waiting for admin approval.', [
        'product_id' => $productId,
        'images' => $uploadedImages
    ], 201);
    
} catch(PDOException $e) {
    sendResponse(false, 'Failed to add product: ' . $e->getMessage(), null, 500);
}

?>

